# -*- coding: utf-8 -*-

import mod.server.extraServerApi as serverApi
import time

ServerSystem = serverApi.GetServerSystemCls()
compFactory = serverApi.GetEngineCompFactory()


# 用来本地调试，设为True时，可以在本地简单测试发货或数据存储等流程
DEBUG = False


class LobbyGoodsServerSystem(ServerSystem):

    def __init__(self, namespace, systemName):
        super(LobbyGoodsServerSystem, self).__init__(namespace, systemName)
        print "===== LobbyGoodsServerSystem init ====="

        if not DEBUG:
            if serverApi.GetPlatform() != -1:
                # 非联机大厅环境下，compFactory.CreateHttp会返回None
                # 如果该组件在联机大厅外也可用，需要对不同平台做不同处理
                print "非联机大厅！"
                return

        self.ListenEvent()
        self.mRefreshShipPlayerIds = {}  # 需要获取订单信息的玩家队列 {玩家id: 剩余重试次数}
        compFactory.CreateGame(serverApi.GetLevelId()).AddRepeatedTimer(4, self.PollShip)  # 4秒轮询一次玩家订单
        # 商品实现指令对应的发货函数
        self.mCmdFunc = {
            'add_money_100': self._AddMoney100
        }
        # 记录玩家的uid
        self.mUid = {}
        # 记录玩家金币
        self.mMoney = {}
        # 记录玩家初始物品，类型为{playerId: {itemName: time}}，time为过期时间
        self.mItems = {}
        # 记录一局游戏内玩家获得的金币
        self.mBattleMoney = {}

    def ListenEvent(self):
        self.ListenForEvent(serverApi.GetEngineNamespace(), serverApi.GetEngineSystemName(), "AddServerPlayerEvent", self, self.OnAddServerPlayer)
        self.ListenForEvent(serverApi.GetEngineNamespace(), serverApi.GetEngineSystemName(), "DelServerPlayerEvent", self, self.OnDelServerPlayer)
        self.ListenForEvent(serverApi.GetEngineNamespace(), serverApi.GetEngineSystemName(), "ServerChatEvent", self, self.OnServerChat)
        self.ListenForEvent(serverApi.GetEngineNamespace(), serverApi.GetEngineSystemName(), "lobbyGoodBuySucServerEvent", self, self.OnLobbyGoodBuySucServerEvent)
        self.ListenForEvent(serverApi.GetEngineNamespace(), serverApi.GetEngineSystemName(), "MobDieEvent", self, self.OnMobDie)

    # 玩家进入战斗时，记录uid
    def OnAddServerPlayer(self, args):
        playerId = args['id']
        comp = compFactory.CreateHttp(serverApi.GetLevelId())
        self.mUid[playerId] = comp.GetPlayerUid(playerId)

    # 玩家离开，移出轮询队列
    def OnDelServerPlayer(self, args):
        self.mRefreshShipPlayerIds.pop(args['id'], None)

    def OnLobbyGoodBuySucServerEvent(self, args):
        playerId = args["eid"]
        if not args['buyItem']:
            # 玩家登录
            comp = compFactory.CreateMsg(playerId)
            comp.NotifyOneMessage(playerId, '输入buysword购买钻石剑，输入endbattle结算战斗，输入top获取金币最多的前10玩家')
            # 请求玩家数据
            self._InitPlayerData(playerId)
            # 请求一次订单信息
            self.mRefreshShipPlayerIds[playerId] = 0
        else:
            # 玩家购买了商品，轮询订单信息，最多重试5次
            self.mRefreshShipPlayerIds[playerId] = 4
        # 立即发起订单查询
        self.ShipPlayer(playerId)

    # 轮询玩家订单信息
    def PollShip(self):
        for playerId in self.mRefreshShipPlayerIds.keys():
            self.ShipPlayer(playerId)

    # 查询某个玩家的订单并发货
    def ShipPlayer(self, playerId):
        retry = self.mRefreshShipPlayerIds[playerId]
        # 1. 先把玩家从轮询队列删除
        del self.mRefreshShipPlayerIds[playerId]

        def callback(data):
            if data:
                orders = data["entity"]["orders"]
                # 没有未发货订单，重试
                if not orders:
                    if retry > 0:
                        # 3. 需要重试的时候再把玩家加回轮询队列
                        self.mRefreshShipPlayerIds[playerId] = retry - 1
                    return
                # 发货订单
                for order in orders:
                    self.mCmdFunc[order['cmd']](order['order_id'], playerId, order['product_count'])
            else:
                # 请求失败
                comp = compFactory.CreateMsg(playerId)
                comp.NotifyOneMessage(playerId, "获取订单失败！")
                return

        # 2. 调用查询订单接口
        if not DEBUG:
            httpComp = compFactory.CreateHttp(serverApi.GetLevelId())
            httpComp.QueryLobbyUserItem(callback, self.mUid[playerId])
        else:
            # 本地构造数据模拟订单发货
            callback({
                'entity': {
                    'orders': [
                        {
                            'order_id': 'x',
                            'cmd': 'add_money_100',
                            'product_count': 1
                        }
                    ]
                }
            })

    # 初始化玩家数据
    def _InitPlayerData(self, playerId):
        def callback(data):
            if data:
                newData = {i["key"]: i["value"] for i in data["entity"]["data"]}
                # 获取的key如果没有设置过，就不会出现在返回的data中
                # 2. 记录玩家金币
                self.mMoney[playerId] = newData.get('money', 0)
                comp = compFactory.CreateMsg(playerId)
                comp.NotifyOneMessage(playerId, "当前金币数量：" + str(self.mMoney[playerId]))
                # 3. 记录玩家初始物品
                self.mItems[playerId] = newData.get('items', {})
                comp.NotifyOneMessage(playerId, "当前初始装备：" + str(self.mItems[playerId].keys()))
                # 3. 给玩家发放初始物品
                self._TryGiveItems(playerId)
            else:
                # 请求失败
                comp = compFactory.CreateMsg(playerId)
                comp.NotifyOneMessage(playerId, "获取数据失败！")

        # 1. 请求玩家金币数量与初始物品
        if not DEBUG:
            httpComp = compFactory.CreateHttp(serverApi.GetLevelId())
            httpComp.LobbyGetStorage(callback, self.mUid[playerId], ['money', 'items'])
        else:
            # 本地构造数据模拟数据获取
            callback({
                'entity': {
                    'data': [
                        {
                            'key': 'money',
                            'value': 100
                        }
                    ]
                }
            })

    # 给玩家添加100金币
    def _AddMoney100(self, orderId, playerId, count):
        def getter():
            return [
                {
                    # 返回当前金币加上100
                    'key': 'money',
                    'value': self.mMoney[playerId] + 100 * count
                }
            ]

        def callback(data):
            if data:
                code = data['code']
                if code == 0 or code == 2 or code == 5:
                    # code为0表示成功，2表示数据冲突，5表示订单冲突
                    # 成功或冲突都需要使用新数据更新本地数据
                    # 获取的key如果没有设置过，就不会出现在返回的data中
                    newData = {i["key"]: i["value"] for i in data["entity"]["data"]}
                    self.mMoney[playerId] = newData.get('money', 0)

                    if code == 0:  # 发货成功
                        comp = compFactory.CreateMsg(playerId)
                        comp.NotifyOneMessage(playerId, "购买成功！当前金币数量：" + str(self.mMoney[playerId]))
                    elif code == 5:  # 重复发货
                        comp = compFactory.CreateMsg(playerId)
                        comp.NotifyOneMessage(playerId, "订单已发货！金币数量：" + str(self.mMoney[playerId]))

                else:
                    # 其他原因失败了
                    comp = compFactory.CreateMsg(playerId)
                    comp.NotifyOneMessage(playerId, "发货失败！")
            else:
                # 请求失败
                comp = compFactory.CreateMsg(playerId)
                comp.NotifyOneMessage(playerId, "发货失败！")

        # 标记订单发货，并添加金币
        if not DEBUG:
            httpComp = compFactory.CreateHttp(serverApi.GetLevelId())
            httpComp.LobbySetStorageAndUserItem(callback, self.mUid[playerId], orderId, getter)
        else:
            # 本地构造模拟数据
            callback({
                'code': 0,
                'entity': {
                    'data': getter()
                }
            })

    # 给玩家发放初始物品
    def _TryGiveItems(self, playerId):
        # 1. 首先从extraData里获取已经发过给玩家的物品
        entitycomp = compFactory.CreateExtraData(playerId)
        existItems = entitycomp.GetExtraData("lobbyGoodsDemo_items")
        if not existItems:
            existItems = {}
        dirty = False
        # 2. 查找没有发过给玩家的，以及在有效期内的物品并发给玩家
        now = time.time()
        for item, expireTime in self.mItems[playerId].iteritems():
            if item not in existItems and now < expireTime:
                itemDict = {
                    'itemName': item,
                    'count': 1,
                }
                comp = compFactory.CreateItem(playerId)
                comp.SpawnItemToPlayerInv(itemDict, playerId, 0)
                dirty = True
        # 3. 如果实际发过物品，则更新到extraData
        if dirty:
            entitycomp.SetExtraData("lobbyGoodsDemo_items", self.mItems[playerId])

    # 使用10金币购买钻石剑，有效期为7天
    def _BuyDiamondSword(self, playerId):
        itemName = 'minecraft:diamond_sword'
        # 检查金币是否足够
        if self.mMoney[playerId] < 10:
            comp = compFactory.CreateMsg(playerId)
            comp.NotifyOneMessage(playerId, "金币不足！")

        def getter():
            # 需要再次检查金币数量，因为getter调用可能是异步的
            if self.mMoney[playerId] < 10:
                comp = compFactory.CreateMsg(playerId)
                comp.NotifyOneMessage(playerId, "金币不足！")
                return None

            # 深拷贝一份数据
            items = dict(self.mItems[playerId])
            # 如果还在有效期内，则有效期加7天，否则有效期为当前时间加7天
            expireTime = items.get(itemName, 0)
            now = time.time()
            if expireTime > now:
                items[itemName] += 7 * 24 * 60 * 60
            else:
                items[itemName] = now + 7 * 24 * 60 * 60
            return [
                # 设置金币为当前金币减10
                {
                    'key': 'money',
                    'value': self.mMoney[playerId] - 10
                },
                {
                    'key': 'items',
                    'value': items
                }
            ]

        def callback(data):
            if data:
                code = data['code']
                if code == 0 or code == 2:
                    # 成功或冲突时，使用新数据更新本地数据
                    # 这里跟订单没关系，所以不会出现订单冲突的情况
                    # 获取的key如果没有设置过，就不会出现在返回的data中
                    newData = {i["key"]: i["value"] for i in data["entity"]["data"]}
                    self.mMoney[playerId] = newData.get('money', 0)
                    self.mItems[playerId] = newData.get('items', {})
                    if code == 0:
                        # 数据设置成功
                        self._TryGiveItems(playerId)
                        comp = compFactory.CreateMsg(playerId)
                        formatTime = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(self.mItems[playerId][itemName]))
                        comp.NotifyOneMessage(playerId, "购买钻石剑成功！有效期至：" + formatTime + "。剩余金币数量：" + str(self.mMoney[playerId]))
                else:
                    # 其他原因失败了
                    comp = compFactory.CreateMsg(playerId)
                    comp.NotifyOneMessage(playerId, "购买失败！")
            else:
                # 请求失败
                comp = compFactory.CreateMsg(playerId)
                comp.NotifyOneMessage(playerId, "购买失败！")

        # 设置数据
        if not DEBUG:
            httpComp = compFactory.CreateHttp(serverApi.GetLevelId())
            httpComp.LobbySetStorageAndUserItem(callback, self.mUid[playerId], None, getter)
        else:
            callback({
                'code': 0,
                'entity': {
                    'data': getter()
                }
            })

    # 上传单个玩家的结算数据
    def _EndBattleForOnePlayer(self, playerId):
        def getter():
            return [
                # 返回已有金币加上本局玩家获得的金币
                {
                    'key': 'money',
                    'value': self.mMoney[playerId] + self.mBattleMoney[playerId]
                }
            ]

        def callback(data):
            if data:
                code = data['code']
                if code == 0 or code == 2:
                    # 成功或冲突时，使用新数据更新本地数据
                    # 这里跟订单没关系，所以不会出现订单冲突的情况
                    # 获取的key如果没有设置过，就不会出现在返回的data中
                    newData = {i["key"]: i["value"] for i in data["entity"]["data"]}
                    self.mMoney[playerId] = newData.get('money', 0)
                    if code == 0:
                        # 数据设置成功
                        comp = compFactory.CreateMsg(playerId)
                        comp.NotifyOneMessage(playerId, "战斗结算成功！当前金币数量：" + str(self.mMoney[playerId]))
                        # 检查是否结算完毕
                        del self.mBattleMoney[playerId]
                        if not self.mBattleMoney:
                            # 结算完毕
                            comp.NotifyOneMessage(playerId, "战斗结算完毕！")
                else:
                    # 其他原因失败了
                    comp = compFactory.CreateMsg(playerId)
                    comp.NotifyOneMessage(playerId, "结算失败！")
            else:
                # 请求失败
                comp = compFactory.CreateMsg(playerId)
                comp.NotifyOneMessage(playerId, "结算失败！")

        # 设置数据
        if not DEBUG:
            httpComp = compFactory.CreateHttp(serverApi.GetLevelId())
            httpComp.LobbySetStorageAndUserItem(callback, self.mUid[playerId], None, getter)
        else:
            callback({
                'code': 0,
                'entity': {
                    'data': getter()
                }
            })

    # 结算一局战斗，上传金币
    # 连续调用多次接口时，一定要注意写法！！！
    # 将callback，getter以及接口的调用封装到一个子函数内！！！
    def _EndBattle(self):
        for playerId in self.mBattleMoney.keys():
            self._EndBattleForOnePlayer(playerId)

        # 错误写法如下，在此函数内调用了多次接口
        '''
        for playerId in self.mBattleMoney.keys():
            def getter():
                return [
                    {
                        'key': 'money',
                        'value': self.mMoney[playerId] + self.mBattleMoney[playerId]
                    }
                ]
            def callback(data):
                print playerId
                ...
            httpComp.LobbySetStorageAndUserItem(callback, self.mUid[playerId], None, getter)
        '''

    # 获取最多金币的前10名玩家
    def _GetTop(self):
        def callback(data):
            if data:
                # 不要用这个接口返回的数据更新self.mMoney!!!
                comp = serverApi.GetEngineCompFactory().CreateGame(serverApi.GetLevelId())
                for i, item in enumerate(data["entity"]["data"]):
                    comp.SetNotifyMsg("No.%s: %s, 金币: %s" % (i+1, item['nickname'], item['value']))
            else:
                # 请求失败
                comp = serverApi.GetEngineCompFactory().CreateGame(serverApi.GetLevelId())
                comp.SetNotifyMsg("获取排行失败！")
        if not DEBUG:
            httpComp = compFactory.CreateHttp(serverApi.GetLevelId())
            httpComp.LobbyGetStorageBySort(callback, 'money', False, 0, 10)
        else:
            callback({
                'code': 0,
                'entity': {
                    'data': [
                        {
                            'nickname': 'Steve',
                            'value': 100
                        }
                    ]
                }
            })

    # 测试指令
    def OnServerChat(self, args):
        if args['message'] == 'buysword':
            self._BuyDiamondSword(args['playerId'])
        elif args['message'] == 'endbattle':
            self._EndBattle()
        elif args['message'] == 'top':
            self._GetTop()
        elif DEBUG and args['message'] == "uiinit":
            # 因为本地测试不会触发lobbyGoodBuySucServerEvent事件，所以用指令代替
            # 您也可以在客户端uiInit事件通知服务端时调用lobbyGoodBuy的回调
            self.OnLobbyGoodBuySucServerEvent({
                'eid': args['playerId'],
                'buyItem': False
            })

    # 玩家每杀一个僵尸获得1金币
    def OnMobDie(self, args):
        dieEntity = args['id']
        if compFactory.CreateEngineType(dieEntity).GetEngineTypeStr() == 'minecraft:zombie':
            playerId = args['attacker']
            if compFactory.CreateEngineType(playerId).GetEngineTypeStr() == 'minecraft:player':
                if playerId not in self.mBattleMoney:
                    self.mBattleMoney[playerId] = 0
                self.mBattleMoney[playerId] += 1

    def Destroy(self):
        print "===== LobbyGoodsServerSystem Destroy ====="
